'use client'

import { useState, useEffect } from 'react'
import { useParams, useRouter } from 'next/navigation'
import { supabase } from '@/lib/supabase'
import { Cliente, Producto } from '@/lib/types'
import Link from 'next/link'

export default function EditarSuscripcionPage() {
  const params = useParams()
  const router = useRouter()
  const [loading, setLoading] = useState(true)
  const [guardando, setGuardando] = useState(false)
  const [clientes, setClientes] = useState<Cliente[]>([])
  const [productos, setProductos] = useState<Producto[]>([])
  const [busquedaCliente, setBusquedaCliente] = useState('')
  const [formData, setFormData] = useState({
    cliente_id: '',
    tipo: 'mensual' as 'mensual' | 'anual',
    precio: '',
    fecha_inicio: '',
    fecha_fin: '',
    patente: '',
    producto_id: '',
    cantidad: '1',
    incluir_iva: false,
    estado: 'activa' as 'activa' | 'vencida' | 'cancelada',
  })
  const [usarFechaFinManual, setUsarFechaFinManual] = useState(false)

  useEffect(() => {
    if (params.id) {
      fetchSuscripcion(params.id as string)
      fetchClientes()
      fetchProductos()
    }
  }, [params.id])

  async function fetchSuscripcion(id: string) {
    try {
      const { data, error } = await supabase
        .from('suscripciones')
        .select('*')
        .eq('id', id)
        .single()

      if (error) throw error
      
      // Asegurarse de que las fechas estén en formato YYYY-MM-DD sin conversión de zona horaria
      const fechaInicio = data.fecha_inicio ? data.fecha_inicio.split('T')[0] : ''
      const fechaFin = data.fecha_fin ? data.fecha_fin.split('T')[0] : ''
      
      setFormData({
        cliente_id: data.cliente_id,
        tipo: data.tipo,
        precio: data.precio.toString(),
        fecha_inicio: fechaInicio,
        fecha_fin: fechaFin || calcularFechaFin(fechaInicio, data.tipo),
        patente: data.patente || '',
        producto_id: data.producto_id || '',
        cantidad: '1', // Por defecto 1, se puede calcular desde precio/producto si existe
        incluir_iva: data.incluir_iva || false,
        estado: data.estado,
      })
      // Si ya existe una fecha_fin, significa que fue ingresada manualmente
      setUsarFechaFinManual(!!data.fecha_fin)
    } catch (error) {
      console.error('Error fetching suscripcion:', error)
    } finally {
      setLoading(false)
    }
  }

  async function fetchClientes() {
    try {
      const { data, error } = await supabase
        .from('clientes')
        .select('*')
        .eq('activo', true)
        .order('nombre')

      if (error) throw error
      setClientes(data || [])
    } catch (error) {
      console.error('Error fetching clientes:', error)
    }
  }

  async function fetchProductos() {
    try {
      const { data, error } = await supabase
        .from('productos')
        .select('*')
        .order('nombre')

      if (error) throw error
      setProductos(data || [])
      console.log('Productos cargados:', data?.length || 0)
    } catch (error) {
      console.error('Error fetching productos:', error)
    }
  }

  function handleProductoChange(productoId: string) {
    const producto = productos.find(p => p.id === productoId)
    const cantidad = parseFloat(formData.cantidad) || 1
    if (producto) {
      const precioTotal = producto.precio * cantidad
      setFormData({
        ...formData,
        producto_id: productoId,
        precio: precioTotal.toString()
      })
    } else {
      setFormData({
        ...formData,
        producto_id: '',
        precio: formData.precio // Mantener el precio actual si se deselecciona
      })
    }
  }

  function handleCantidadChange(cantidad: string) {
    const cantidadNum = parseFloat(cantidad) || 1
    setFormData({
      ...formData,
      cantidad: cantidad
    })
    
    // Si hay un producto seleccionado, recalcular el precio
    if (formData.producto_id) {
      const producto = productos.find(p => p.id === formData.producto_id)
      if (producto) {
        const precioTotal = producto.precio * cantidadNum
        setFormData(prev => ({
          ...prev,
          cantidad: cantidad,
          precio: precioTotal.toString()
        }))
      }
    }
  }

  function calcularFechaFin(fechaInicio: string, tipo: string): string {
    // Trabajar directamente con strings YYYY-MM-DD para evitar problemas de zona horaria
    const [year, month, day] = fechaInicio.split('-').map(Number)
    
    let nuevoYear = year
    let nuevoMonth = month
    let nuevoDay = day
    
    if (tipo === 'anual') {
      nuevoYear = year + 1
    } else {
      nuevoMonth = month + 1
      // Ajustar si el mes pasa de 12
      if (nuevoMonth > 12) {
        nuevoMonth = 1
        nuevoYear = year + 1
      }
    }
    
    // Formatear con ceros a la izquierda
    const mesStr = nuevoMonth.toString().padStart(2, '0')
    const diaStr = nuevoDay.toString().padStart(2, '0')
    
    return `${nuevoYear}-${mesStr}-${diaStr}`
  }

  function handleFechaInicioChange(fechaInicio: string) {
    setFormData({ 
      ...formData, 
      fecha_inicio: fechaInicio,
      fecha_fin: usarFechaFinManual ? formData.fecha_fin : calcularFechaFin(fechaInicio, formData.tipo)
    })
  }

  function handleTipoChange(tipo: 'mensual' | 'anual') {
    setFormData({ 
      ...formData, 
      tipo: tipo,
      fecha_fin: usarFechaFinManual ? formData.fecha_fin : (formData.fecha_inicio ? calcularFechaFin(formData.fecha_inicio, tipo) : '')
    })
  }

  async function handleSubmit(e: React.FormEvent) {
    e.preventDefault()
    setGuardando(true)

    try {
      const { error } = await supabase
        .from('suscripciones')
        .update({
          cliente_id: formData.cliente_id,
          tipo: formData.tipo,
          precio: parseFloat(formData.precio),
          fecha_inicio: formData.fecha_inicio,
          fecha_fin: usarFechaFinManual && formData.fecha_fin ? formData.fecha_fin : (formData.fecha_fin || calcularFechaFin(formData.fecha_inicio, formData.tipo)),
          patente: formData.patente || null,
          producto_id: formData.producto_id || null,
          incluir_iva: formData.incluir_iva,
          estado: formData.estado,
        })
        .eq('id', params.id)

      if (error) throw error

      router.push(`/suscripciones/${params.id}`)
    } catch (error: any) {
      console.error('Error updating suscripcion:', error)
      alert(`Error al actualizar la suscripción: ${error.message}`)
    } finally {
      setGuardando(false)
    }
  }

  if (loading) {
    return <div className="text-center p-8">Cargando...</div>
  }

  return (
    <div className="max-w-2xl mx-auto">
      <div className="mb-4">
        <Link href={`/suscripciones/${params.id}`} className="text-blue-600 hover:underline">← Volver</Link>
      </div>
      <h1 className="text-3xl font-bold mb-6">Editar Suscripción</h1>
      <form onSubmit={handleSubmit} className="bg-white p-6 rounded-lg shadow space-y-4">
        <div>
          <label className="block text-sm font-medium mb-1">Cliente *</label>
          <input
            type="text"
            placeholder="🔍 Buscar cliente por nombre, empresa, email o teléfono..."
            value={busquedaCliente}
            onChange={(e) => setBusquedaCliente(e.target.value)}
            className="w-full border rounded px-3 py-2 mb-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
          />
          <select
            required
            value={formData.cliente_id}
            onChange={(e) => {
              setFormData({ ...formData, cliente_id: e.target.value })
              // Limpiar búsqueda al seleccionar
              setBusquedaCliente('')
            }}
            className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
          >
            <option value="">Selecciona un cliente</option>
            {(() => {
              const clientesFiltrados = clientes.filter((cliente) => {
                if (!busquedaCliente) return true
                const termino = busquedaCliente.toLowerCase()
                return (
                  cliente.nombre?.toLowerCase().includes(termino) ||
                  cliente.empresa?.toLowerCase().includes(termino) ||
                  cliente.email?.toLowerCase().includes(termino) ||
                  cliente.telefono?.toLowerCase().includes(termino) ||
                  cliente.rfc?.toLowerCase().includes(termino)
                )
              })
              return clientesFiltrados.map((cliente) => (
                <option key={cliente.id} value={cliente.id}>
                  {cliente.nombre || cliente.empresa || 'Cliente sin nombre'} {cliente.nombre && cliente.empresa ? `- ${cliente.empresa}` : ''}
                </option>
              ))
            })()}
          </select>
          {busquedaCliente && (() => {
            const clientesFiltrados = clientes.filter((cliente) => {
              const termino = busquedaCliente.toLowerCase()
              return (
                cliente.nombre?.toLowerCase().includes(termino) ||
                cliente.empresa?.toLowerCase().includes(termino) ||
                cliente.email?.toLowerCase().includes(termino) ||
                cliente.telefono?.toLowerCase().includes(termino) ||
                cliente.rfc?.toLowerCase().includes(termino)
              )
            })
            return (
              <p className="text-xs text-gray-500 mt-1">
                {clientesFiltrados.length === 0 
                  ? 'No se encontraron clientes'
                  : `${clientesFiltrados.length} cliente(s) encontrado(s)`
                }
              </p>
            )
          })()}
        </div>

        <div>
          <label className="block text-sm font-medium mb-1">Tipo de Suscripción *</label>
          <select
            required
            value={formData.tipo}
            onChange={(e) => handleTipoChange(e.target.value as 'mensual' | 'anual')}
            className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
          >
            <option value="mensual">Mensual</option>
            <option value="anual">Anual</option>
          </select>
        </div>

        <div>
          <label className="block text-sm font-medium mb-1">Producto o Servicio</label>
          <select
            value={formData.producto_id}
            onChange={(e) => handleProductoChange(e.target.value)}
            className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
          >
            <option value="">Selecciona un producto o servicio (opcional)</option>
            {productos.length === 0 ? (
              <option value="" disabled>No hay productos disponibles. Crea productos en la sección "Productos"</option>
            ) : (
              productos.map((producto) => (
                <option key={producto.id} value={producto.id}>
                  {producto.nombre} - ${producto.precio.toLocaleString()} {!producto.activo ? '(Inactivo)' : ''}
                </option>
              ))
            )}
          </select>
          <p className="text-xs text-gray-500 mt-1">
            {productos.length === 0 
              ? 'No hay productos creados. Ve a "Productos" para crear uno.'
              : `Al seleccionar un producto, el precio se rellenará automáticamente. ${productos.length} producto(s) disponible(s).`
            }
          </p>
        </div>

        {formData.producto_id && (
          <div>
            <label className="block text-sm font-medium mb-1">Cantidad *</label>
            <input
              type="number"
              required
              min="1"
              step="1"
              value={formData.cantidad}
              onChange={(e) => handleCantidadChange(e.target.value)}
              className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
            />
            <p className="text-xs text-gray-500 mt-1">El precio se calculará automáticamente (precio unitario × cantidad)</p>
          </div>
        )}

        <div>
          <label className="block text-sm font-medium mb-1">Precio *</label>
          <input
            type="number"
            required
            step="0.01"
            min="0"
            value={formData.precio}
            onChange={(e) => setFormData({ ...formData, precio: e.target.value })}
            className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
            placeholder="0.00"
          />
        </div>

        <div className="flex items-center">
          <input
            type="checkbox"
            id="incluir_iva"
            checked={formData.incluir_iva}
            onChange={(e) => setFormData({ ...formData, incluir_iva: e.target.checked })}
            className="w-4 h-4 text-blue-600 border-gray-300 rounded focus:ring-blue-500"
          />
          <label htmlFor="incluir_iva" className="ml-2 text-sm font-medium text-gray-700">
            Incluir IVA (19%)
          </label>
        </div>

        <div>
          <label className="block text-sm font-medium mb-1">Fecha de Inicio *</label>
          <input
            type="date"
            required
            value={formData.fecha_inicio}
            onChange={(e) => handleFechaInicioChange(e.target.value)}
            className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
          />
        </div>

        <div>
          <label className="block text-sm font-medium mb-1">Patente / Dispositivo</label>
          <input
            type="text"
            value={formData.patente}
            onChange={(e) => setFormData({ ...formData, patente: e.target.value })}
            className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
            placeholder="Ej: ABC123, GPS-001, etc."
          />
          <p className="text-xs text-gray-500 mt-1">Identificador del dispositivo GPS asociado a esta suscripción</p>
        </div>

        <div>
          <label className="block text-sm font-medium mb-1">Fecha de Fin</label>
          <input
            type="date"
            value={formData.fecha_fin || calcularFechaFin(formData.fecha_inicio, formData.tipo)}
            onChange={(e) => {
              setFormData({ ...formData, fecha_fin: e.target.value })
              setUsarFechaFinManual(true)
            }}
            className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
          />
          {formData.fecha_inicio && (
            <div className="mt-2 flex items-center gap-2">
              <p className="text-sm text-gray-600">
                <strong>Calculada automáticamente:</strong>{' '}
                {calcularFechaFin(formData.fecha_inicio, formData.tipo)}
                {' '}({formData.tipo === 'anual' ? '1 año' : '1 mes'} después del inicio)
              </p>
              {usarFechaFinManual && formData.fecha_fin && (
                <button
                  type="button"
                  onClick={() => {
                    setFormData({ 
                      ...formData, 
                      fecha_fin: calcularFechaFin(formData.fecha_inicio, formData.tipo) 
                    })
                    setUsarFechaFinManual(false)
                  }}
                  className="text-sm text-blue-600 hover:underline"
                >
                  Usar fecha calculada
                </button>
              )}
            </div>
          )}
        </div>

        <div>
          <label className="block text-sm font-medium mb-1">Estado *</label>
          <select
            required
            value={formData.estado}
            onChange={(e) => setFormData({ ...formData, estado: e.target.value as any })}
            className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
          >
            <option value="activa">Activa</option>
            <option value="vencida">Vencida</option>
            <option value="cancelada">Cancelada</option>
          </select>
        </div>

        <div className="flex gap-4 pt-4">
          <button
            type="submit"
            disabled={guardando}
            className="bg-blue-600 text-white px-6 py-2 rounded hover:bg-blue-700 disabled:opacity-50 transition"
          >
            {guardando ? 'Guardando...' : 'Guardar Cambios'}
          </button>
          <button
            type="button"
            onClick={() => router.back()}
            className="bg-gray-300 text-gray-700 px-6 py-2 rounded hover:bg-gray-400 transition"
          >
            Cancelar
          </button>
        </div>
      </form>
    </div>
  )
}
